# Copyright 2024 The Chromium Authors
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.
"""Codegen for Java types used to appease the compiler. .class file is discarded.

Needed to avoid ciruclar dependencies in the build graph.
"""


def Generate(java_class,
             nested_classes,
             script_name,
             proxy_interface=None,
             proxy_natives=None):
  """Generate an empty placeholder java class with a given name and package.

  The placeholder files allow compiling FooJni.java without requiring access to
  Foo.java or any of its deps/imports.
  """
  sb = []
  sb.append(f"""\
//
// This file is an empty placeholder. It was generated by {script_name}
//

package {java_class.package_with_dots};

public interface {java_class.name} {{
""")
  if nested_classes:
    for clazz in nested_classes:
      # The proxy interface has to exist for real in the placeholder file for
      # Foo.java because FooJni implements it and uses @Override.
      if clazz == proxy_interface:
        sb.append(f'  public interface {clazz.nested_name} {{\n')
        for native in proxy_natives:
          parameter_list = ', '.join(f'{param.java_type.to_java()} {param.name}'
                                     for param in native.params)
          sb.append(
              f'    public {native.return_type.to_java()} {native.name}({parameter_list});\n'
          )
        sb.append('  }\n')
      else:
        sb.append(f'  public interface {clazz.nested_name} {{}}\n')

  sb.append('}\n')
  return ''.join(sb)
