// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// This file contains types/functions and constants for platform handle wrapping
// and unwrapping APIs.
//
// Note: This header should be compilable as C.

#ifndef MOJO_PUBLIC_C_SYSTEM_PLATFORM_HANDLE_H_
#define MOJO_PUBLIC_C_SYSTEM_PLATFORM_HANDLE_H_

#include <stdint.h>

#include "mojo/public/c/system/system_export.h"
#include "mojo/public/c/system/types.h"

#ifdef __cplusplus
extern "C" {
#endif

// |MojoPlatformHandleType|: A value indicating the specific type of platform
//     handle encapsulated by a MojoPlatformHandle (see below.) This is stored
//     in the MojoPlatformHandle's |type| field and determines how the |value|
//     field is interpreted.
//
//   |MOJO_PLATFORM_HANDLE_TYPE_INVALID| - An invalid platform handle.
//   |MOJO_PLATFORM_HANDLE_TYPE_FILE_DESCRIPTOR| - A file descriptor. Only valid
//       on POSIX systems.
//   |MOJO_PLATFORM_HANDLE_TYPE_MACH_PORT| - A Mach port. Only valid on OS X.
//   |MOJO_PLATFORM_HANDLE_TYPE_WINDOWS_HANDLE| - A Windows HANDLE value. Only
//       valid on Windows.
//   |MOJO_PLATFORM_HANDLE_TYPE_FUCHSIA_HANDLE| - A Fuchsia zx_handle_t value.
//       Only valid on Fuchsia.

typedef uint32_t MojoPlatformHandleType;

#ifdef __cplusplus
const MojoPlatformHandleType MOJO_PLATFORM_HANDLE_TYPE_INVALID = 0;
const MojoPlatformHandleType MOJO_PLATFORM_HANDLE_TYPE_FILE_DESCRIPTOR = 1;
const MojoPlatformHandleType MOJO_PLATFORM_HANDLE_TYPE_MACH_PORT = 2;
const MojoPlatformHandleType MOJO_PLATFORM_HANDLE_TYPE_WINDOWS_HANDLE = 3;
const MojoPlatformHandleType MOJO_PLATFORM_HANDLE_TYPE_FUCHSIA_HANDLE = 4;
#else
#define MOJO_PLATFORM_HANDLE_TYPE_INVALID ((MojoPlatformHandleType)0)
#define MOJO_PLATFORM_HANDLE_TYPE_FILE_DESCRIPTOR ((MojoPlatformHandleType)1)
#define MOJO_PLATFORM_HANDLE_TYPE_MACH_PORT ((MojoPlatformHandleType)2)
#define MOJO_PLATFORM_HANDLE_TYPE_WINDOWS_HANDLE ((MojoPlatformHandleType)3)
#define MOJO_PLATFORM_HANDLE_TYPE_FUCHSIA_HANDLE ((MojoPlatformHandleType)4)
#endif

// |MojoPlatformHandle|: A handle to a native platform object.
//
//     |uint32_t struct_size|: The size of this structure. Used for versioning
//         to allow for future extensions.
//
//     |MojoPlatformHandleType type|: The type of handle stored in |value|.
//
//     |uint64_t value|: The value of this handle. Ignored if |type| is
//         MOJO_PLATFORM_HANDLE_TYPE_INVALID. Otherwise the meaning of this
//         value depends on the value of |type|.
//

struct MOJO_ALIGNAS(8) MojoPlatformHandle {
  uint32_t struct_size;
  MojoPlatformHandleType type;
  uint64_t value;
};
MOJO_STATIC_ASSERT(sizeof(MojoPlatformHandle) == 16,
                   "MojoPlatformHandle has wrong size");

// |MojoSharedBufferGuid|: A GUID used to identify the buffer backing a shared
//     buffer handle.
struct MOJO_ALIGNAS(8) MojoSharedBufferGuid {
  uint64_t high;
  uint64_t low;
};

// |MojoPlatformSharedBufferHandleFlags|: Flags relevant to wrapped platform
//     shared buffers.
//
//   |MOJO_PLATFORM_SHARED_BUFFER_HANDLE_FLAG_NONE| - No flags.
//   |MOJO_PLATFORM_SHARED_BUFFER_HANDLE_FLAG_HANDLE_IS_READ_ONLY| - Indicates
//       that the handle being wrapped is mappable only to read-only memory.
//
//       IMPORTANT: THIS DOES NOT CONTROL MEMORY PROTECTION ON THE WRAPPED
//       HANDLE. It merely conveys a |MojoWrapPlatformSharedBufferHandle()|
//       caller's knowledge about whether or not the handle being wrapped is
//       already read-only.

typedef uint32_t MojoPlatformSharedBufferHandleFlags;

#ifdef __cplusplus
const MojoPlatformSharedBufferHandleFlags
    MOJO_PLATFORM_SHARED_BUFFER_HANDLE_FLAG_NONE = 0;

const MojoPlatformSharedBufferHandleFlags
    MOJO_PLATFORM_SHARED_BUFFER_HANDLE_FLAG_HANDLE_IS_READ_ONLY = 1 << 0;
#else
#define MOJO_PLATFORM_SHARED_BUFFER_HANDLE_FLAG_NONE \
  ((MojoPlatformSharedBufferHandleFlags)0)

#define MOJO_PLATFORM_SHARED_BUFFER_HANDLE_FLAG_HANDLE_IS_READ_ONLY \
  ((MojoPlatformSharedBufferHandleFlags)1 << 0)
#endif

// Wraps a native platform handle as a Mojo handle which can be transferred
// over a message pipe. Takes ownership of the underlying native platform
// object.
//
// |platform_handle|: The platform handle to wrap.
//
// Returns:
//     |MOJO_RESULT_OK| if the handle was successfully wrapped. In this case
//         |*mojo_handle| contains the Mojo handle of the wrapped object.
//     |MOJO_RESULT_RESOURCE_EXHAUSTED| if the system is out of handles.
//     |MOJO_RESULT_INVALID_ARGUMENT| if |platform_handle| was not a valid
//          platform handle.
//
// NOTE: It is not always possible to detect if |platform_handle| is valid,
// particularly when |platform_handle->type| is valid but
// |platform_handle->value| does not represent a valid platform object.
MOJO_SYSTEM_EXPORT MojoResult
MojoWrapPlatformHandle(const struct MojoPlatformHandle* platform_handle,
                       MojoHandle* mojo_handle);  // Out

// Unwraps a native platform handle from a Mojo handle. If this call succeeds,
// ownership of the underlying platform object is assumed by the caller. The
// The Mojo handle is always closed regardless of success or failure.
//
// |mojo_handle|: The Mojo handle from which to unwrap the native platform
//     handle.
//
// Returns:
//     |MOJO_RESULT_OK| if the handle was successfully unwrapped. In this case
//         |*platform_handle| contains the unwrapped platform handle.
//     |MOJO_RESULT_INVALID_ARGUMENT| if |mojo_handle| was not a valid Mojo
//         handle wrapping a platform handle.
MOJO_SYSTEM_EXPORT MojoResult
MojoUnwrapPlatformHandle(MojoHandle mojo_handle,
                         struct MojoPlatformHandle* platform_handle);  // Out

// Wraps a native platform shared buffer handle as a Mojo shared buffer handle
// which can be used exactly like a shared buffer handle created by
// |MojoCreateSharedBuffer()| or |MojoDuplicateBufferHandle()|.
//
// Takes ownership of the native platform shared buffer handle.
//
// |platform_handle|: The platform handle to wrap. Must be a native handle to a
//     shared buffer object.
// |num_bytes|: The size of the shared buffer in bytes.
// |flags|: Flags which influence the treatment of the shared buffer object. See
//     the |MojoPlatformSharedBufferHandleFlags| definition for details.
//
// Returns:
//     |MOJO_RESULT_OK| if the handle was successfully wrapped. In this case
//         |*mojo_handle| contains a Mojo shared buffer handle.
//     |MOJO_RESULT_INVALID_ARGUMENT| if |platform_handle| was not a valid
//         platform shared buffer handle.
MOJO_SYSTEM_EXPORT MojoResult MojoWrapPlatformSharedBufferHandle(
    const struct MojoPlatformHandle* platform_handle,
    size_t num_bytes,
    const struct MojoSharedBufferGuid* guid,
    MojoPlatformSharedBufferHandleFlags flags,
    MojoHandle* mojo_handle);  // Out

// Unwraps a native platform shared buffer handle from a Mojo shared buffer
// handle. If this call succeeds, ownership of the underlying shared buffer
// object is assumed by the caller.
//
// The Mojo handle is always closed regardless of success or failure.
//
// |mojo_handle|: The Mojo shared buffer handle to unwrap.
//
// |platform_handle|, |num_bytes| and |flags| are used to receive output values
// and MUST always be non-null.
//
// Returns:
//    |MOJO_RESULT_OK| if the handle was successfully unwrapped. In this case
//        |*platform_handle| contains a platform shared buffer handle,
//        |*num_bytes| contains the size of the shared buffer object, and
//        |*flags| indicates flags relevant to the wrapped buffer (see below).
//    |MOJO_RESULT_INVALID_ARGUMENT| if |mojo_handle| is not a valid Mojo
//        shared buffer handle.
//
// Flags which may be set in |*flags| upon success:
//    |MOJO_PLATFORM_SHARED_BUFFER_HANDLE_FLAG_HANDLE_IS_READ_ONLY| is set iff
//        the unwrapped shared buffer handle can only be mapped to a read-only
//        memory segment.
MOJO_SYSTEM_EXPORT MojoResult MojoUnwrapPlatformSharedBufferHandle(
    MojoHandle mojo_handle,
    struct MojoPlatformHandle* platform_handle,
    size_t* num_bytes,
    struct MojoSharedBufferGuid* guid,
    MojoPlatformSharedBufferHandleFlags* flags);

#ifdef __cplusplus
}  // extern "C"
#endif

#endif  // MOJO_PUBLIC_C_SYSTEM_PLATFORM_HANDLE_H_
