// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef UI_EVENTS_EVENT_CONSTANTS_H_
#define UI_EVENTS_EVENT_CONSTANTS_H_

#include "build/build_config.h"

namespace ui {

// Event flags currently supported. It is OK to add values to the middle of
// this list and/or reorder it, but make sure you also touch the various other
// enums/constants that want to stay in sync with this. For example,
// KeyEventFlags and MouseEventFlags should not overlap EventFlags.
enum EventFlags {
  EF_NONE = 0,  // Used to denote no flags explicitly

  // Universally applicable status bits.
  EF_IS_SYNTHESIZED = 1 << 0,

  // Modifier key state.
  EF_SHIFT_DOWN = 1 << 1,
  EF_CONTROL_DOWN = 1 << 2,
  EF_ALT_DOWN = 1 << 3,
  EF_COMMAND_DOWN = 1 << 4,   // GUI Key (e.g. Command on OS X
                              // keyboards, Search on Chromebook
                              // keyboards, Windows on MS-oriented
                              // keyboards)
  EF_FUNCTION_DOWN = 1 << 5,  // Function key.
  EF_ALTGR_DOWN = 1 << 6,
  EF_MOD3_DOWN = 1 << 7,

  // Other keyboard states.
  EF_NUM_LOCK_ON = 1 << 8,
  EF_CAPS_LOCK_ON = 1 << 9,
  EF_SCROLL_LOCK_ON = 1 << 10,

  // Mouse buttons.
  EF_LEFT_MOUSE_BUTTON = 1 << 11,
  EF_MIDDLE_MOUSE_BUTTON = 1 << 12,
  EF_RIGHT_MOUSE_BUTTON = 1 << 13,
  EF_BACK_MOUSE_BUTTON = 1 << 14,
  EF_FORWARD_MOUSE_BUTTON = 1 << 15,
  EF_MOUSE_BUTTON = EF_LEFT_MOUSE_BUTTON | EF_MIDDLE_MOUSE_BUTTON |
                    EF_RIGHT_MOUSE_BUTTON | EF_BACK_MOUSE_BUTTON |
                    EF_FORWARD_MOUSE_BUTTON,

// An artificial value used to bridge platform differences.
// Many commands on Mac as Cmd+Key are the counterparts of
// Ctrl+Key on other platforms.
#if BUILDFLAG(IS_APPLE)
  EF_PLATFORM_ACCELERATOR = EF_COMMAND_DOWN,
#else
  EF_PLATFORM_ACCELERATOR = EF_CONTROL_DOWN,
#endif
};

// Flags specific to key events.
// WARNING: If you add or remove values make sure traits for serializing these
// values are updated.
enum KeyEventFlags {
  EF_IME_FABRICATED_KEY = 1 << 16,  // Key event fabricated by the underlying
                                    // IME without a user action.
                                    // (Linux X11 only)
  EF_IS_REPEAT = 1 << 17,
  EF_FINAL = 1 << 18,             // Do not remap; the event was created with
                                  // the desired final values.
  EF_IS_EXTENDED_KEY = 1 << 19,   // Windows extended key (see WM_KEYDOWN doc)
  EF_IS_STYLUS_BUTTON = 1 << 20,  // Event was generated by a stylus button
  EF_MAX_KEY_EVENT_FLAGS_VALUE = (1 << 21) - 1,
};

// Flags specific to mouse events.
enum MouseEventFlags {
  EF_IS_DOUBLE_CLICK = 1 << 16,
  EF_IS_TRIPLE_CLICK = 1 << 17,
  EF_IS_NON_CLIENT = 1 << 18,
  EF_FROM_TOUCH = 1 << 19,           // Indicates this mouse event is generated
                                     // from an unconsumed touch/gesture event.
  EF_TOUCH_ACCESSIBILITY = 1 << 20,  // Indicates this event was generated from
                                     // touch accessibility mode.
  EF_CURSOR_HIDE = 1 << 21,          // Indicates this mouse event is generated
                                     // because the cursor was just hidden. This
                                     // can be used to update hover state.
  EF_PRECISION_SCROLLING_DELTA =     // Indicates this mouse event is from high
  1 << 22,                           // precision touchpad and will come with a
                                     // high precision delta.
  EF_SCROLL_BY_PAGE = 1 << 23,       // Indicates this mouse event is generated
                                     // when users is requesting to scroll by
                                     // pages.
  EF_UNADJUSTED_MOUSE = 1 << 24,     // Indicates this mouse event is unadjusted
                                     // mouse events that has unadjusted
                                     // movement delta, i.e. is from WM_INPUT
                                     // on Windows.
  EF_NOT_SUITABLE_FOR_MOUSE_WARPING =  // Indicates this mouse event should not
  1 << 25,                             // trigger mouse warping (which moves the
                                       // mouse to another display when the
                                       // mouse hits the window boundaries).
};

// Result of dispatching an event.
enum EventResult {
  ER_UNHANDLED = 0,      // The event hasn't been handled. The event can be
                         // propagated to other handlers.
  ER_HANDLED = 1 << 0,   // The event has already been handled, but it can
                         // still be propagated to other handlers.
  ER_CONSUMED = 1 << 1,  // The event has been handled, and it should not be
                         // propagated to other handlers.
  ER_DISABLE_SYNC_HANDLING =
      1 << 2,  // The event shouldn't be handled synchronously. This
               // happens if the event is being handled
               // asynchronously, or if the event is invalid and
               // shouldn't be handled at all.
};

// Phase of the event dispatch.
enum EventPhase {
  EP_PREDISPATCH,
  EP_PRETARGET,
  EP_TARGET,
  EP_POSTTARGET,
  EP_POSTDISPATCH
};

// Phase information used for a ScrollEvent. ScrollEventPhase is for scroll
// stream from user gesture, EventMomentumPhase is for inertia scroll stream
// after user gesture.
enum class ScrollEventPhase {
  // Event has no phase information. eg. the Event is not in a scroll stream.
  kNone,

  // Event is the beginning of a scroll event stream.
  kBegan,

  // Event is a scroll event with phase information.
  kUpdate,

  // Event is the end of the current scroll event stream.
  kEnd,
};

// Momentum phase information used for a ScrollEvent.
enum class EventMomentumPhase {
  // Event is a non-momentum update to an event stream already begun.
  NONE,

  // Event is the beginning of an event stream that may result in momentum.
  // BEGAN vs MAY_BEGIN:
  // - BEGAN means we already know the inertia scroll stream must happen after
  //   BEGAN event. On Windows touchpad, we sent this when receive the first
  //   inertia scroll event or Direct Manipulation state change to INERTIA.
  // - MAY_BEGIN means the inertia scroll stream may happen after MAY_BEGIN
  //   event. On Mac, we send this when receive releaseTouches, but we do not
  //   know the inertia scroll stream will happen or not at that time.
  BEGAN,

  // Event maybe the beginning of an event stream that may result in momentum.
  // This state used on Mac.
  MAY_BEGIN,

  // Event is an update while in a momentum phase. A "begin" event for the
  // momentum phase portion of an event stream uses this also, but the scroll
  // offsets will be zero.
  INERTIAL_UPDATE,

  // Event marks the end of the current event stream. Note that this is also set
  // for events that are not a "stream", but indicate both the start and end of
  // the event (e.g. a mouse wheel tick).
  END,

  // EventMomentumPhase can only be BLOCKED when ScrollEventPhase is kEnd. Event
  // marks the end of the current event stream, when there will be no inertia
  // scrolling after the user gesture. ScrollEventPhase must simultaneously be
  // kEnd because that is when it is determined if an event stream that results
  // in momentum will begin or not. This phase is only used on Windows.
  BLOCKED,
};

enum EventDeviceId {
  // Device ID for Touch, Mouse and Key Events.
  ED_UNKNOWN_DEVICE = -1,
  // Device ID for events injected through a remote connection (like CRD).
  ED_REMOTE_INPUT_DEVICE = -2,
};

// Pointing device type.
enum class EventPointerType : int {
  kUnknown,
  kMouse,
  kPen,
  kTouch,
  kEraser,
  kMaxValue = kEraser,
};

// Device type for gesture events.
enum class GestureDeviceType : int {
  DEVICE_UNKNOWN = 0,
  DEVICE_TOUCHPAD,
  DEVICE_TOUCHSCREEN,
};

}  // namespace ui

#endif  // UI_EVENTS_EVENT_CONSTANTS_H_
