// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef UI_WM_CORE_WINDOW_MODALITY_CONTROLLER_H_
#define UI_WM_CORE_WINDOW_MODALITY_CONTROLLER_H_

#include <vector>

#include "base/compiler_specific.h"
#include "ui/aura/env_observer.h"
#include "ui/aura/window_observer.h"
#include "ui/events/event_handler.h"
#include "ui/wm/wm_export.h"

namespace ui {
class EventTarget;
class LocatedEvent;
}

namespace wm {

// Sets the modal parent for the child.
WM_EXPORT void SetModalParent(aura::Window* child, aura::Window* parent);

// Returns the modal transient child of |window|, or NULL if |window| does not
// have any modal transient children.
WM_EXPORT aura::Window* GetModalTransient(aura::Window* window);

// WindowModalityController is an event filter that consumes events sent to
// windows that are the transient parents of window-modal windows. This filter
// must be added to the CompoundEventFilter so that activation works properly.
class WM_EXPORT WindowModalityController : public ui::EventHandler,
                                           public aura::EnvObserver,
                                           public aura::WindowObserver {
 public:
  explicit WindowModalityController(ui::EventTarget* event_target);
  virtual ~WindowModalityController();

  // Overridden from ui::EventHandler:
  virtual void OnKeyEvent(ui::KeyEvent* event) OVERRIDE;
  virtual void OnMouseEvent(ui::MouseEvent* event) OVERRIDE;
  virtual void OnTouchEvent(ui::TouchEvent* event) OVERRIDE;

  // Overridden from aura::EnvObserver:
  virtual void OnWindowInitialized(aura::Window* window) OVERRIDE;

  // Overridden from aura::WindowObserver:
  virtual void OnWindowPropertyChanged(aura::Window* window,
                                       const void* key,
                                       intptr_t old) OVERRIDE;
  virtual void OnWindowVisibilityChanged(aura::Window* window,
                                         bool visible) OVERRIDE;
  virtual void OnWindowDestroyed(aura::Window* window) OVERRIDE;

 private:
  // Processes a mouse/touch event, and returns true if the event should be
  // consumed.
  bool ProcessLocatedEvent(aura::Window* target,
                           ui::LocatedEvent* event);

  std::vector<aura::Window*> windows_;

  ui::EventTarget* event_target_;

  DISALLOW_COPY_AND_ASSIGN(WindowModalityController);
};

}  // namespace wm

#endif  // UI_WM_CORE_WINDOW_MODALITY_CONTROLLER_H_
