// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <algorithm>
#include <iostream>

#include "base/memory/scoped_ptr.h"
#include "base/sys_byteorder.h"
#include "net/spdy/spdy_frame_reader.h"
#include "testing/platform_test.h"

namespace net {

TEST(SpdyFrameReaderTest, ReadUInt16) {
  // Frame data in network byte order.
  const uint16 kFrameData[] = {
    htons(1), htons(1<<15),
  };

  SpdyFrameReader frame_reader(reinterpret_cast<const char *>(kFrameData),
                               arraysize(kFrameData) * sizeof(uint16));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  uint16 uint16_val;
  EXPECT_TRUE(frame_reader.ReadUInt16(&uint16_val));
  EXPECT_FALSE(frame_reader.IsDoneReading());
  EXPECT_EQ(1, uint16_val);

  EXPECT_TRUE(frame_reader.ReadUInt16(&uint16_val));
  EXPECT_TRUE(frame_reader.IsDoneReading());
  EXPECT_EQ(1<<15, uint16_val);
}

TEST(SpdyFrameReaderTest, ReadUInt32) {
  // Frame data in network byte order.
  const uint32 kFrameData[] = {
    htonl(1), htonl(0x80000000),
  };

  SpdyFrameReader frame_reader(reinterpret_cast<const char *>(kFrameData),
                               arraysize(kFrameData) * sizeof(uint32));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  uint32 uint32_val;
  EXPECT_TRUE(frame_reader.ReadUInt32(&uint32_val));
  EXPECT_FALSE(frame_reader.IsDoneReading());
  EXPECT_EQ(1u, uint32_val);

  EXPECT_TRUE(frame_reader.ReadUInt32(&uint32_val));
  EXPECT_TRUE(frame_reader.IsDoneReading());
  EXPECT_EQ(1u<<31, uint32_val);
}

TEST(SpdyFrameReaderTest, ReadStringPiece16) {
  // Frame data in network byte order.
  const char kFrameData[] = {
    0x00, 0x02,  // uint16(2)
    0x48, 0x69,  // "Hi"
    0x00, 0x10,  // uint16(16)
    0x54, 0x65, 0x73, 0x74,
    0x69, 0x6e, 0x67, 0x2c,
    0x20, 0x31, 0x2c, 0x20,
    0x32, 0x2c, 0x20, 0x33,  // "Testing, 1, 2, 3"
  };

  SpdyFrameReader frame_reader(kFrameData, arraysize(kFrameData));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  base::StringPiece stringpiece_val;
  EXPECT_TRUE(frame_reader.ReadStringPiece16(&stringpiece_val));
  EXPECT_FALSE(frame_reader.IsDoneReading());
  EXPECT_EQ(0, stringpiece_val.compare("Hi"));

  EXPECT_TRUE(frame_reader.ReadStringPiece16(&stringpiece_val));
  EXPECT_TRUE(frame_reader.IsDoneReading());
  EXPECT_EQ(0, stringpiece_val.compare("Testing, 1, 2, 3"));
}

TEST(SpdyFrameReaderTest, ReadStringPiece32) {
  // Frame data in network byte order.
  const char kFrameData[] = {
    0x00, 0x00, 0x00, 0x03,  // uint32(3)
    0x66, 0x6f, 0x6f,  // "foo"
    0x00, 0x00, 0x00, 0x10,  // uint32(16)
    0x54, 0x65, 0x73, 0x74,
    0x69, 0x6e, 0x67, 0x2c,
    0x20, 0x34, 0x2c, 0x20,
    0x35, 0x2c, 0x20, 0x36,  // "Testing, 4, 5, 6"
  };

  SpdyFrameReader frame_reader(kFrameData, arraysize(kFrameData));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  base::StringPiece stringpiece_val;
  EXPECT_TRUE(frame_reader.ReadStringPiece32(&stringpiece_val));
  EXPECT_FALSE(frame_reader.IsDoneReading());
  EXPECT_EQ(0, stringpiece_val.compare("foo"));

  EXPECT_TRUE(frame_reader.ReadStringPiece32(&stringpiece_val));
  EXPECT_TRUE(frame_reader.IsDoneReading());
  EXPECT_EQ(0, stringpiece_val.compare("Testing, 4, 5, 6"));
}

TEST(SpdyFrameReaderTest, ReadUInt16WithBufferTooSmall) {
  // Frame data in network byte order.
  const char kFrameData[] = {
    0x00,  // part of a uint16
  };

  SpdyFrameReader frame_reader(kFrameData, arraysize(kFrameData));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  uint16 uint16_val;
  EXPECT_FALSE(frame_reader.ReadUInt16(&uint16_val));
}

TEST(SpdyFrameReaderTest, ReadUInt32WithBufferTooSmall) {
  // Frame data in network byte order.
  const char kFrameData[] = {
    0x00, 0x00, 0x00,  // part of a uint32
  };

  SpdyFrameReader frame_reader(kFrameData, arraysize(kFrameData));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  uint32 uint32_val;
  EXPECT_FALSE(frame_reader.ReadUInt32(&uint32_val));

  // Also make sure that trying to read a uint16, which technically could work,
  // fails immediately due to previously encountered failed read.
  uint16 uint16_val;
  EXPECT_FALSE(frame_reader.ReadUInt16(&uint16_val));
}

// Tests ReadStringPiece16() with a buffer too small to fit the entire string.
TEST(SpdyFrameReaderTest, ReadStringPiece16WithBufferTooSmall) {
  // Frame data in network byte order.
  const char kFrameData[] = {
    0x00, 0x03,  // uint16(3)
    0x48, 0x69,  // "Hi"
  };

  SpdyFrameReader frame_reader(kFrameData, arraysize(kFrameData));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  base::StringPiece stringpiece_val;
  EXPECT_FALSE(frame_reader.ReadStringPiece16(&stringpiece_val));

  // Also make sure that trying to read a uint16, which technically could work,
  // fails immediately due to previously encountered failed read.
  uint16 uint16_val;
  EXPECT_FALSE(frame_reader.ReadUInt16(&uint16_val));
}

// Tests ReadStringPiece16() with a buffer too small even to fit the length.
TEST(SpdyFrameReaderTest, ReadStringPiece16WithBufferWayTooSmall) {
  // Frame data in network byte order.
  const char kFrameData[] = {
    0x00,  // part of a uint16
  };

  SpdyFrameReader frame_reader(kFrameData, arraysize(kFrameData));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  base::StringPiece stringpiece_val;
  EXPECT_FALSE(frame_reader.ReadStringPiece16(&stringpiece_val));

  // Also make sure that trying to read a uint16, which technically could work,
  // fails immediately due to previously encountered failed read.
  uint16 uint16_val;
  EXPECT_FALSE(frame_reader.ReadUInt16(&uint16_val));
}

// Tests ReadStringPiece32() with a buffer too small to fit the entire string.
TEST(SpdyFrameReaderTest, ReadStringPiece32WithBufferTooSmall) {
  // Frame data in network byte order.
  const char kFrameData[] = {
    0x00, 0x00, 0x00, 0x03,  // uint32(3)
    0x48, 0x69,  // "Hi"
  };

  SpdyFrameReader frame_reader(kFrameData, arraysize(kFrameData));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  base::StringPiece stringpiece_val;
  EXPECT_FALSE(frame_reader.ReadStringPiece32(&stringpiece_val));

  // Also make sure that trying to read a uint16, which technically could work,
  // fails immediately due to previously encountered failed read.
  uint16 uint16_val;
  EXPECT_FALSE(frame_reader.ReadUInt16(&uint16_val));
}

// Tests ReadStringPiece32() with a buffer too small even to fit the length.
TEST(SpdyFrameReaderTest, ReadStringPiece32WithBufferWayTooSmall) {
  // Frame data in network byte order.
  const char kFrameData[] = {
    0x00, 0x00, 0x00,  // part of a uint32
  };

  SpdyFrameReader frame_reader(kFrameData, arraysize(kFrameData));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  base::StringPiece stringpiece_val;
  EXPECT_FALSE(frame_reader.ReadStringPiece32(&stringpiece_val));

  // Also make sure that trying to read a uint16, which technically could work,
  // fails immediately due to previously encountered failed read.
  uint16 uint16_val;
  EXPECT_FALSE(frame_reader.ReadUInt16(&uint16_val));
}

TEST(SpdyFrameReaderTest, ReadBytes) {
  // Frame data in network byte order.
  const char kFrameData[] = {
    0x66, 0x6f, 0x6f,  // "foo"
    0x48, 0x69,  // "Hi"
  };

  SpdyFrameReader frame_reader(kFrameData, arraysize(kFrameData));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  char dest1[3] = {};
  EXPECT_TRUE(frame_reader.ReadBytes(&dest1, arraysize(dest1)));
  EXPECT_FALSE(frame_reader.IsDoneReading());
  EXPECT_EQ("foo", base::StringPiece(dest1, arraysize(dest1)));

  char dest2[2] = {};
  EXPECT_TRUE(frame_reader.ReadBytes(&dest2, arraysize(dest2)));
  EXPECT_TRUE(frame_reader.IsDoneReading());
  EXPECT_EQ("Hi", base::StringPiece(dest2, arraysize(dest2)));
}

TEST(SpdyFrameReaderTest, ReadBytesWithBufferTooSmall) {
  // Frame data in network byte order.
  const char kFrameData[] = {
    0x01,
  };

  SpdyFrameReader frame_reader(kFrameData, arraysize(kFrameData));
  EXPECT_FALSE(frame_reader.IsDoneReading());

  char dest[arraysize(kFrameData) + 2] = {};
  EXPECT_FALSE(frame_reader.ReadBytes(&dest, arraysize(kFrameData) + 1));
  EXPECT_STREQ("", dest);
}

}  // namespace net
