// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef UI_ACCEROMETER_ACCELEROMETER_TYPES_H_
#define UI_ACCEROMETER_ACCELEROMETER_TYPES_H_

#include "base/macros.h"
#include "ui/accelerometer/ui_accelerometer_export.h"
#include "ui/gfx/geometry/vector3d_f.h"

namespace ui {

enum AccelerometerSource {
  // Accelerometer is located in the device's screen. In the screen's natural
  // orientation, positive X points to the right, consistent with the pixel
  // position. Positive Y points up the screen. Positive Z is perpendicular to
  // the screen, pointing outwards towards the user. The orientation is
  // described at:
  // http://www.html5rocks.com/en/tutorials/device/orientation/.
  ACCELEROMETER_SOURCE_SCREEN = 0,

  // Accelerometer is located in a keyboard attached to the device's screen.
  // If the device is open 180 degrees the orientation is consistent with the
  // screen. I.e. Positive X points to the right, positive Y points up on the
  // keyboard and positive Z is perpendicular to the keyboard pointing out
  // towards the user.
  ACCELEROMETER_SOURCE_ATTACHED_KEYBOARD,

  ACCELEROMETER_SOURCE_COUNT
};

struct UI_ACCELEROMETER_EXPORT AccelerometerReading {
  AccelerometerReading();
  ~AccelerometerReading();

  // If true, this accelerometer is being updated.
  bool present;

  // The reading from this accelerometer measured in m/s^2.
  gfx::Vector3dF reading;
};

// An accelerometer update contains the last known value for each of the
// accelerometers present on the device.
class UI_ACCELEROMETER_EXPORT AccelerometerUpdate {
 public:
  AccelerometerUpdate();
  ~AccelerometerUpdate();

  // Returns true if |source| has a valid value in this update.
  bool has(AccelerometerSource source)  const {
    return data_[source].present;
  }

  // Returns the last known value for |source|.
  const gfx::Vector3dF& get(AccelerometerSource source) const {
    return data_[source].reading;
  }

  void Set(AccelerometerSource source, float x, float y, float z) {
    data_[source].present = true;
    data_[source].reading.set_x(x);
    data_[source].reading.set_y(y);
    data_[source].reading.set_z(z);
  }

 protected:
  AccelerometerReading data_[ACCELEROMETER_SOURCE_COUNT];

  DISALLOW_COPY_AND_ASSIGN(AccelerometerUpdate);
};

}  // namespace chromeos

#endif  // UI_ACCEROMETER_ACCELEROMETER_TYPES_H_
