/*
 * Copyright (C) 2010 Google Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "core/html/forms/URLInputType.h"

#include "core/InputTypeNames.h"
#include "core/html/HTMLInputElement.h"
#include "core/html/parser/HTMLParserIdioms.h"
#include "platform/text/PlatformLocale.h"

namespace blink {

InputType* URLInputType::create(HTMLInputElement& element)
{
    return new URLInputType(element);
}

void URLInputType::countUsage()
{
    countUsageIfVisible(UseCounter::InputTypeURL);
}

const AtomicString& URLInputType::formControlType() const
{
    return InputTypeNames::url;
}

bool URLInputType::typeMismatchFor(const String& value) const
{
    return !value.isEmpty() && !KURL(KURL(), value).isValid();
}

bool URLInputType::typeMismatch() const
{
    return typeMismatchFor(element().value());
}

String URLInputType::typeMismatchText() const
{
    return locale().queryString(WebLocalizedString::ValidationTypeMismatchForURL);
}

String URLInputType::sanitizeValue(const String& proposedValue) const
{
    return BaseTextInputType::sanitizeValue(stripLeadingAndTrailingHTMLSpaces(proposedValue));
}

String URLInputType::sanitizeUserInputValue(const String& proposedValue) const
{
    // Do not call URLInputType::sanitizeValue.
    return BaseTextInputType::sanitizeValue(proposedValue);
}

} // namespace blink
