/****************************************************************************
**
** Copyright (C) 2018 The Qt Company Ltd.
** Contact: http://www.qt.io/licensing/
**
** This file is part of Qt 3D Studio.
**
** $QT_BEGIN_LICENSE:GPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 or (at your option) any later version
** approved by the KDE Free Qt Foundation. The licenses are as published by
** the Free Software Foundation and appearing in the file LICENSE.GPL3
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#ifndef Q3DSOBJECT3D_P_H
#define Q3DSOBJECT3D_P_H

//
//  W A R N I N G
//  -------------
//
// This file is not part of the Qt API.  It exists purely as an
// implementation detail.  This header file may change from version to
// version without notice, or even be removed.
//
// We mean it.
//

#include <Qt3DStudioRuntime2/private/q3dsruntimeglobal_p.h>
#include <QQuickItem>
#include <private/q3dsuippresentation_p.h>

QT_BEGIN_NAMESPACE

class Q3DSLayer3D;

class Q3DSV_PRIVATE_EXPORT Q3DSObject3D : public QQuickItem
{
    Q_OBJECT

public:
    Q3DSObject3D();
    ~Q3DSObject3D();

    virtual Q3DSGraphObject *createObject(Q3DSUipPresentation *presentation) = 0;

    Q3DSGraphObject *object() const { return m_object; }
    Q3DSLayer3D *layer() const { return m_layer; }

    enum InvalidateFlag {
        InvalidateObject = 0x01,
        InvalidateLayer = 0x02
    };
    Q_DECLARE_FLAGS(InvalidateFlags, InvalidateFlag)

    void invalidate(InvalidateFlags flags);

protected:
    void itemChange(ItemChange, const ItemChangeData &) override;
    QByteArray makeIdAndName() const;
    virtual void propertyChanged(const QMetaProperty &sp,
                                 Q3DSGraphObject *target,
                                 const QMetaProperty &tp);

private:
    void activate();
    void activate_helper(Q3DSLayer3D *layer);
    void setLayer_recursive(Q3DSLayer3D *layer);
    void syncProperties();

    Q3DSGraphObject *m_object = nullptr;
    Q3DSLayer3D *m_layer = nullptr;
    Q3DSObject3D *m_pendingObjectParent = nullptr;
    bool m_hasPendingActivate = false;
    QMetaObject::Connection m_layerConn;

    friend class Q3DSLayer3D;
    friend class Q3DSPropertyBinder;
};

Q_DECLARE_OPERATORS_FOR_FLAGS(Q3DSObject3D::InvalidateFlags)

class Q3DSPropertyBinder : public QObject
{
public:
    ~Q3DSPropertyBinder() { unbind(); }
    static QByteArray binderName() { return QByteArrayLiteral("_q3dsbinding"); }
    static QByteArray propertyPrefix() { return QByteArrayLiteral("qq_"); }

    Q3DSPropertyBinder(QObject *p = nullptr) : QObject(p) {}
    int qt_metacall(QMetaObject::Call call, int methodId, void **args) override
    {
        methodId = QObject::qt_metacall(call, methodId, args);
        if (methodId < 0)
            return methodId;

        if (!m_dst || !m_src)
            return  methodId;

        if (call == QMetaObject::InvokeMetaMethod) {
            QMetaProperty sp = m_src->metaObject()->property(methodId);
            const int dstIdx = m_propMap[methodId];
            QMetaProperty tp = (dstIdx >= 0) ? m_dst->metaObject()->property(dstIdx)
                                             : QMetaProperty();
            m_src->propertyChanged(sp, m_dst, tp);
            return -1;
        }
        return methodId;
    }

    void bind(Q3DSObject3D *src, Q3DSGraphObject *dst)
    {
        if (src == m_src && dst == m_dst)
            return;

        unbind();

        m_src = src;
        m_src->setProperty(binderName(), QVariant::fromValue(this));
        m_dst = dst;
        m_dst->setProperty(binderName(), QVariant::fromValue(this));
        const int propCount = src->metaObject()->propertyCount();
        for (int i = 0; i != propCount; ++i) {
            const auto property = src->metaObject()->property(i);
            if (!property.hasNotifySignal())
                continue;
            const QByteArray name = propertyPrefix() + property.name();
            int idx = dst->metaObject()->indexOfProperty(name.constData());
            if (idx < 0)
                idx = dst->metaObject()->indexOfProperty(name.constData() + 3);
            m_propMap[i] = idx;
            QMetaObject::connect(src, property.notifySignalIndex(), this, i + metaObject()->methodCount(), Qt::DirectConnection, nullptr);
        }
    }

    Q_INVOKABLE void unbind()
    {
        if (m_src) {
            const QVariant h = m_src->property(binderName());
            Q_ASSERT(this == h.value<Q3DSPropertyBinder *>());
            m_src->setProperty(binderName(), QVariant());
        }

        if (m_dst) {
            const QVariant h = m_dst->property(binderName());
            Q_ASSERT(this == h.value<Q3DSPropertyBinder *>());
            m_dst->setProperty(binderName(), QVariant());
        }

        disconnect();
        m_propMap.clear();
        m_src = nullptr;
        m_dst = nullptr;
    }

private:
    QMap<int, int> m_propMap;
    Q3DSObject3D *m_src = nullptr;
    Q3DSGraphObject *m_dst = nullptr;
};

QT_END_NAMESPACE

#endif // Q3DSOBJECT3D_P_H
